<?php
// This script is intended to be included within admin_dashboard.php,
// ensuring $conn (database connection) is available.

// NOTE: The previous safety check for $conn has been removed to address
// the "Headers already sent" warning (image_552cc9.png/image_551686.png).
// We rely on the parent file (admin_dashboard.php) to properly initialize $conn.

$message = '';
$message_class = '';

// --- 1. HANDLE DOCUMENT UPLOAD LOGIC ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['upload_document'])) {
    
    // Check if a file was actually uploaded
    if (isset($_FILES['document_file']) && $_FILES['document_file']['error'] === UPLOAD_ERR_OK) {
        
        $file_info = $_FILES['document_file'];
        $file_name_raw = $file_info['name'];
        $file_extension = strtolower(pathinfo($file_name_raw, PATHINFO_EXTENSION));
        
        // **File Type Validation: Allowed Document Types**
        $allowed_extensions = ['pdf', 'doc', 'docx', 'xlsx', 'xls', 'txt', 'ppt', 'pptx', 'rtf', 'csv'];
        if (!in_array($file_extension, $allowed_extensions)) {
            $message = "Error: File type '{$file_extension}' is not allowed. Only common document formats (PDF, DOCX, XLSX, etc.) are supported.";
            $message_class = 'error';
        } else {
            // Use the raw filename without extension as the document name
            $document_name = pathinfo($file_name_raw, PATHINFO_FILENAME);

            // Define the server path for documents
            $file_path = 'uploads/general_documents/' . $file_name_raw; // Placeholder path
            
            // Use the authenticated user's ID for the Foreign Key constraint.
            $uploaded_by_id = $_SESSION['user_id'] ?? 1; 

            // **Database Table: general_documents**
            // Columns used: document_id, file_name, file_path, file_extension, uploaded_by, uploaded_at
            $sql_insert = "INSERT INTO general_documents (file_name, file_path, file_extension, uploaded_by) VALUES (?, ?, ?, ?)";

            if (isset($conn) && $stmt = $conn->prepare($sql_insert)) {
                // Bind parameters (string, string, string, integer)
                $stmt->bind_param("sssi", $file_name_raw, $file_path, $file_extension, $uploaded_by_id);
                
                if ($stmt->execute()) {
                    // Placeholder for moving the actual file (DISABLED for this environment)
                    // In a real environment, uncomment the line below:
                    // move_uploaded_file($file_info['tmp_name'], $file_path);
                    
                    $message = "Success: Document '{$file_name_raw}' uploaded successfully (Placeholder).";
                    $message_class = 'success';
                } else {
                    $message = "Error uploading document (DB): " . $stmt->error;
                    $message_class = 'error';
                }
                $stmt->close();
            } else {
                $message = "Error: Database preparation failed for upload or connection unavailable.";
                $message_class = 'error';
            }
        }
    } else {
        // Handle case where no file was selected or a specific upload error occurred
        if (isset($_POST['upload_document'])) {
            $error_code = $_FILES['document_file']['error'] ?? UPLOAD_ERR_NO_FILE;
            
            switch ($error_code) {
                case UPLOAD_ERR_INI_SIZE:
                case UPLOAD_ERR_FORM_SIZE:
                    $message = "Error: The uploaded file exceeds the maximum file size allowed by the server configuration (php.ini limits).";
                    break;
                case UPLOAD_ERR_NO_FILE:
                    $message = "Error: Please select a file to upload.";
                    break;
                default:
                    $message = "Error: An unknown upload error occurred (Code: {$error_code}).";
                    break;
            }
            $message_class = 'error';
        }
    }
}

// --- 2. FETCH ALL DOCUMENTS FOR DISPLAY ---
$all_documents = [];

if (isset($conn)) {
    // Joining to get the username for display
    // IMPORTANT FIX: Using gd.file_name and gd.uploaded_at to match the uploaded table structure (image_5f318d.png)
    $sql_fetch = "
        SELECT 
            gd.document_id, 
            gd.file_name, 
            gd.file_extension, 
            u.username AS uploaded_by_name, 
            gd.uploaded_at 
        FROM general_documents gd
        LEFT JOIN users u ON gd.uploaded_by = u.user_id
        ORDER BY gd.uploaded_at DESC
    ";
    
    $result = $conn->query($sql_fetch);

    if ($result) {
        while ($row = $result->fetch_assoc()) {
            // Extract the simple name (without extension) for display, as done in the previous photo code
            $row['display_name'] = pathinfo($row['file_name'], PATHINFO_FILENAME);
            $row['uploaded_by'] = $row['uploaded_by_name'] ?? 'N/A';
            $all_documents[] = $row;
        }
    } else {
        // This is where the error from image_549e82.png came from previously
        $message_fetch = "Error retrieving documents. Check your database column names: " . $conn->error;
    }
}
?>

<style>
    /* Styling reused from manage_photos.php for consistency */
    .section-header {
        margin-bottom: 25px;
        color: #2c3e50;
    }
    .data-container {
        margin-top: 30px;
    }
    .btn-delete {
        background-color: #e74c3c;
        color: white;
        padding: 5px 10px;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        transition: background-color 0.3s;
    }
    .btn-delete:hover {
        background-color: #c0392b;
    }
    .message {
        padding: 10px;
        margin-bottom: 20px;
        border-radius: 4px;
        font-weight: bold;
    }
    .message.success {
        background-color: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }
    .message.error {
        background-color: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }
    .upload-form-group {
        display: flex;
        gap: 10px;
        align-items: flex-end;
    }
    .upload-form-group > div {
        flex-grow: 1;
    }
    .upload-form-group button {
        margin-top: 0;
        padding: 10px 15px;
    }
</style>

<div class="form-card">
    <h2 class="section-header">Upload New General Document</h2>
    
    <?php if ($message): ?>
        <div class="message <?php echo $message_class; ?>">
            <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>

    <form method="POST" action="admin_dashboard.php?page=manage_documents" enctype="multipart/form-data">
        <input type="hidden" name="upload_document" value="1">
        
        <div class="upload-form-group">
            <div>
                <label for="document_file">Select Document (PDF, DOCX, XLSX, etc.)</label>
                <!-- accept attribute limits browser selection, but server validation is critical -->
                <input type="file" id="document_file" name="document_file" style="padding-top: 5px;" accept=".pdf,.doc,.docx,.xlsx,.xls,.txt,.ppt,.pptx,.rtf,.csv" required>
            </div>
            
            <button type="submit" class="btn-primary">Upload Document</button>
        </div>
    </form>
</div>

<div class="form-card data-container">
    <h2 class="section-header">Document List</h2>
    
    <?php if (isset($message_fetch)): ?>
        <div class="message error"><?php echo htmlspecialchars($message_fetch); ?></div>
    <?php elseif (!empty($all_documents)): ?>
        <table class="data-table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Name</th>
                    <th>Extension</th> 
                    <th>Uploaded By</th>
                    <th>Date</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($all_documents as $document): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($document['document_id']); ?></td>
                        <!-- Display the cleaned name -->
                        <td><?php echo htmlspecialchars($document['display_name']); ?></td> 
                        <td><?php echo htmlspecialchars($document['file_extension']); ?></td>
                        <td><?php echo htmlspecialchars($document['uploaded_by']); ?></td>
                        <td><?php echo htmlspecialchars(substr($document['uploaded_at'], 0, 10)); ?></td>
                        <td>
                            <!-- Action buttons would go here -->
                            <button class="btn-delete">Delete</button>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php else: ?>
        <p>No documents found in the database. Upload one above to get started.</p>
    <?php endif; ?>
</div>
